<?php

namespace App\Helpers;

use Elibyy\TCPDF\Facades\TCPDF as TCPDFFacade;
use App\Helpers\Catalogos;

class PdfInforme extends \TCPDF
{
    private $titulo;
    private $subtitulo;
    private $codigo;
    
    /**
     * Constructor - Configura el PDF
     */
    // Orientación: P=Portrait, L=Landscape
    public function __construct($titulo = 'Informe', $subtitulo = '', $orientation = 'P', $unit = 'mm', $format = 'A4', $codigo = '000000')
    {
        parent::__construct($orientation, $unit, $format, true, 'UTF-8', false);
        
        $this->titulo = $titulo;
        $this->subtitulo = $subtitulo;
        $this->codigo = $codigo;
        
        // Configuración del documento
        $this->SetCreator(PDF_CREATOR);
        $this->SetAuthor('Laburo.com.bo');
        $this->SetTitle($titulo);
        $this->SetSubject($subtitulo);
        
        // Configuración de márgenes
        $this->SetMargins(15, 60, 15); // Izquierda, Superior (70mm para evitar solapamiento), Derecha
        $this->SetHeaderMargin(10);
        $this->SetFooterMargin(15);
        
        // Auto page breaks
        $this->SetAutoPageBreak(true, 25);
        
        // Configuración de fuente
        $this->SetFont('helvetica', '', 10);
        
        // Eliminar línea del header por defecto
        $this->setPrintHeader(true);
        $this->setPrintFooter(true);
    }
    
    /**
     * Header personalizado
     */
    /*
    public function Header()
    {
        // Logo (si existe)
        $logoPath = public_path('assets/images/logo.png');
        if (file_exists($logoPath)) {
            $this->Image($logoPath, 15, 10, 30, '', 'PNG', '', 'T', false, 300, '', false, false, 0, false, false, false);
        }
        
        // Título
        $this->SetFont('helvetica', 'B', 16);
        $this->SetTextColor(33, 37, 41); // Color gris oscuro
        $this->Cell(0, 10, $this->titulo, 0, 1, 'C', 0, '', 0);
        
        // Subtítulo
        if (!empty($this->subtitulo)) {
            $this->SetFont('helvetica', '', 12);
            $this->SetTextColor(108, 117, 125); // Color gris
            $this->Cell(0, 5, $this->subtitulo, 0, 1, 'C', 0, '', 0);
        }
        
        // Fecha y hora
        $this->SetFont('helvetica', 'I', 8);
        $this->SetTextColor(108, 117, 125);
        $this->Cell(0, 5, 'Generado: ' . date('d/m/Y H:i:s'), 0, 1, 'C', 0, '', 0);
        
        // Línea separadora
        $this->SetLineStyle(['width' => 0.5, 'color' => [200, 200, 200]]);
        $this->Line(15, $this->GetY() + 2, $this->getPageWidth() - 15, $this->GetY() + 2);
        
        $this->Ln(5);
    }
    */
    public function Header()
    {
        // Subtítulo
        if (!empty($this->subtitulo)) {
            $this->SetFont('helvetica', '', 9);
            $this->SetTextColor(108, 117, 125); // Color gris
            //$this->Cell(0, 5,'OXG Glasfaser GmbH · Ferdinand-Braun-Platz 1 · D-40549 Düsseldorf', 0, 1, 'L', 0, '', 0);
            // Escribir "OXG" en negrilla
            $this->SetFont('helvetica', 'B', 8);
            $this->Write(0, 'Glasfaser GmbH · '); // recuerda el espacio al final

            // Volver a fuente normal para el resto del texto
            $this->SetFont('helvetica', '', 8);
            $this->Write(0, 'Ferdinand-Braun-Platz 1 · D-40549 Düsseldorf');

            // Salto de línea para continuar debajo si es necesario
            $this->Ln(5);
        }

        // Logo (si existe)
        $logoPath = public_path('assets/images/logo_informe.png');
        if (file_exists($logoPath)) {
            $this->Image($logoPath, 15, 15, 30, '', 'PNG', '', 'T', false, 300, '', false, false, 0, false, false, false);
        }
        
        $this->SetXY(15, 25);

        // Título
        $this->SetFont('helvetica', 'B', 16);
        $this->SetTextColor(33, 37, 41); // Color gris oscuro
        $this->Cell(0, 40, $this->titulo, 0, 1, 'L', 0, '', 0);
    }
    /**
     * Footer personalizado
     */
    public function Footer()
    {
        // Posición a 15 mm del final
        $this->SetY(-15);
        
        // Línea separadora
        $this->SetLineStyle(['width' => 0.5, 'color' => [200, 200, 200]]);
        $this->Line(15, $this->GetY(), $this->getPageWidth() - 15, $this->GetY());
        
        $this->Ln(2);
        
        // Número de página
        $this->SetFont('helvetica', 'I', 8);
        $this->SetTextColor(108, 117, 125);
        if(isset($this->codigo) && $this->codigo != '000000'){
            $this->Cell(0, 5, 'Código: ' . $this->codigo, 0, 0, 'L');
        }
        $this->Cell(0, 5, 'Página ' . $this->getAliasNumPage() . '/' . $this->getAliasNbPages(), 0, 0, 'R');
    }
    
    /**
     * Método auxiliar para agregar una sección con título
     */
    public function agregarSeccion($titulo, $contenido)
    {
        // Título de sección
        $this->SetFont('helvetica', 'B', 12);
        $this->SetTextColor(33, 50, 41);
        $this->SetFillColor(248, 249, 250); // Fondo gris claro
        $this->Cell(0, 8, $titulo, 0, 1, 'L', true);
        $this->Ln(2);
        
        // Contenido
        $this->SetFont('helvetica', '', 10);
        $this->SetTextColor(33, 37, 41);
        $this->MultiCell(0, 5, $contenido, 0, 'L', false, 1);
        $this->Ln(3);
    }
    
    /**
     * Método auxiliar para agregar una tabla
     */
    public function agregarTabla($headers, $data, $widths = null)
    {
        // Calcular anchos automáticamente si no se proporcionan
        if ($widths === null) {
            $numCols = count($headers);
            $totalWidth = $this->getPageWidth() - $this->getMargins()['left'] - $this->getMargins()['right'];
            $widths = array_fill(0, $numCols, $totalWidth / $numCols);
        }
        
        // Headers
        $this->SetFont('helvetica', 'B', 10);
        $this->SetFillColor(52, 58, 64); // Fondo oscuro
        $this->SetTextColor(255, 255, 255); // Texto blanco
        
        foreach ($headers as $i => $header) {
            $this->Cell($widths[$i], 7, $header, 1, 0, 'C', true);
        }
        $this->Ln();
        
        // Datos
        $this->SetFont('helvetica', '', 9);
        $this->SetTextColor(33, 37, 41);
        
        $fill = false;
        foreach ($data as $row) {
            $this->SetFillColor(248, 249, 250); // Fondo gris claro para filas alternas
            foreach ($row as $i => $cell) {
                $this->Cell($widths[$i], 6, $cell, 1, 0, 'L', $fill);
            }
            $this->Ln();
            $fill = !$fill;
        }
        
        $this->Ln(3);
    }

    /**
     * Método auxiliar para agregar una tabla
     */
    public function agregarTabla2($data, $widths = null, $border = 1)
    {
        // Calcular anchos automáticamente si no se proporcionan
        if ($widths === null) {
            // Nota: Se asume que la primera fila define el número de columnas si no hay headers
            $numCols = count($data[0]); 
            $totalWidth = $this->getPageWidth() - $this->getMargins()['left'] - $this->getMargins()['right'];
            $widths = array_fill(0, $numCols, $totalWidth / $numCols);
        }
        
        // Configuración por defecto
        $defaultFontSize = 9;
        $defaultTextColor = [33, 37, 41];
        
        $this->SetFont('helvetica', '', $defaultFontSize);
        $this->SetTextColorArray($defaultTextColor);
        
        $fill = false;
        foreach ($data as $row) {
            // 1. Calcular la altura máxima de la fila
            $maxLines = 1;
            foreach ($row as $i => $cell) {
                // Normalizar celda
                $text = is_array($cell) ? ($cell['text'] ?? '') : $cell;
                $isBold = is_array($cell) && !empty($cell['bold']);
                $fontSize = is_array($cell) && isset($cell['size']) ? $cell['size'] : $defaultFontSize;
                
                // Configurar fuente temporalmente para calcular líneas
                $style = $isBold ? 'B' : '';
                $this->SetFont('helvetica', $style, $fontSize);
                
                // NbLines calcula cuántas líneas ocupará el texto con el ancho dado
                $lines = $this->getNumLines($text, $widths[$i]);
                if ($lines > $maxLines) {
                    $maxLines = $lines;
                }
            }
            
            // Restaurar fuente base para cálculos
            $this->SetFont('helvetica', '', $defaultFontSize);
            
            $rowHeight = 6 * $maxLines; // 6 es la altura base de línea

            // Verificar salto de página si la fila no cabe
            if ($this->GetY() + $rowHeight > $this->getPageHeight() - $this->getMargins()['bottom']) {
                $this->AddPage();
            }

            // 2. Dibujar las celdas
            $startX = $this->GetX();
            $startY = $this->GetY();
            
            $this->SetFillColor(248, 249, 250); // Fondo gris claro para filas alternas
            
            foreach ($row as $i => $cell) {
                // Guardar posición X actual para movernos después
                $currentX = $this->GetX();
                
                // Extraer opciones
                $text = is_array($cell) ? ($cell['text'] ?? '') : $cell;
                $isBold = is_array($cell) && !empty($cell['bold']);
                $align = is_array($cell) && isset($cell['align']) ? $cell['align'] : 'L';
                $fontSize = is_array($cell) && isset($cell['size']) ? $cell['size'] : $defaultFontSize;
                $color = is_array($cell) && isset($cell['color']) ? $cell['color'] : $defaultTextColor;

                // Aplicar estilos específicos de la celda
                $style = $isBold ? 'B' : '';
                $this->SetFont('helvetica', $style, $fontSize);
                $this->SetTextColorArray($color);
                
                // MultiCell(w, h, txt, border, align, fill, ln, x, y, reseth, stretch, ishtml, autopadding, maxh, valign, fitcell)
                $this->MultiCell($widths[$i], $rowHeight, $text, $border, $align, $fill, 0, '', '', true, 0, false, true, $rowHeight, 'M');
                
                // Mover cursor a la derecha para la siguiente celda
                $this->SetXY($currentX + $widths[$i], $startY);
            }
            
            // Restaurar estilos por defecto para la siguiente fila/celda (importante para iteraciones)
            $this->SetFont('helvetica', '', $defaultFontSize);
            $this->SetTextColorArray($defaultTextColor);
            
            // Mover cursor a la siguiente línea (altura máxima de esta fila)
            $this->SetXY($startX, $startY + $rowHeight);
            
            $fill = !$fill;
        }
        
        $this->Ln(3);
    }

    /**
     * Dibuja un cuadro de información con diseño específico
     */
    public function agregarCuadroInformacion($direccion, $sdu, $mdu, $num)
    {
        
        // Posición y tamaño
        $startX = 15;
        $currY = $this->GetY() + 5; // Un poco de espacio antes del cuadro
        $boxWidth = 180;
        $boxHeight = 22; // Altura fija aproximada para 2 líneas de contenido
        $radius = 1;

        // Estilos del cuadro
        $this->SetFillColor(213, 228, 231); // Celeste
        $this->SetDrawColor(213, 228, 231);
        $this->SetLineWidth(0.1);
        
        // Dibujar fondo
        $this->RoundedRect($startX, $currY, $boxWidth, $boxHeight, $radius, '1111', 'DF');
        
        // Configurar posición interna
        $this->SetXY($startX + 4, $currY + 4);
        
        // Color de texto
        $this->SetTextColor(33, 37, 41);

        // --- FILA 1: Dirección ---
        // Etiqueta
        $this->SetFont('helvetica', 'B', 10);
        $this->Cell(20, 6, 'Adresse', 0, 0, 'L', false);
        
        // Valor
        $this->SetFont('helvetica', '', 10);
        $this->Cell(0, 6, $direccion, 0, 1, 'C', false);
        
        // --- FILA 2: Ciudad, Teléfono, Email ---
        $this->SetX($startX + 4); // Reset X al margen interno
        
        // SDU
        $this->SetFont('helvetica', 'B', 10);
        $this->Cell(46, 6, 'Einfamilienhaus (SDU)', 0, 0, 'L', false);
        $this->SetFont('dejavusans', '', 12);
        $this->Cell(10, 6, $sdu, 0, 0, 'L', false);
        
        // MDU
        $this->SetFont('helvetica', 'B', 10);
        $this->Cell(50, 6, 'Mehrfamilienhaus (MDU)', 0, 0, 'L', false);
        $this->SetFont('dejavusans', '', 12);
        $this->Cell(10, 6, $mdu, 0, 0, 'L', false);
        
        // NUM
        $this->SetFont('helvetica', 'B', 10);
        $this->Cell(45, 6, 'Anzahl Wohneinheiten:', 0, 0, 'L', false);
        $this->SetFont('helvetica', '', 10);
        $this->Cell(10, 6, $num, 0, 0, 'L', false);

        // Espacio después del cuadro
        $this->SetY($currY + $boxHeight + 5);
    }

    /**
     * Dibuja la sección de detalles con imagen y 3 columnas
     */
    public function agregarSeccionDetalles($imgData, $col2Data, $col3Data)
    {
        // Configuración de dimensiones
        $startX = 15;
        $boxWidth = 180;
        $boxHeight = 70; // Altura estimada para contener todo
        $radius = 2;
        
        // Verificar espacio disponible en la página
        if ($this->GetY() + $boxHeight > $this->getPageHeight() - $this->getMargins()['bottom']) {
            $this->AddPage();
        }
        
        $currY = $this->GetY() + 5;
        
        // --- 1. Dibujar el Recuadro ---
        $this->SetFillColor(213, 228, 231);
        $this->SetDrawColor(213, 228, 231);
        $this->SetLineWidth(0.1);
        $this->RoundedRect($startX, $currY, $boxWidth, $boxHeight, $radius, '1111', 'DF');
        
        // --- 2. Configurar Columnas ---
        // Anchos aproximados: Col1 (Img) ~30%, Col2 ~35%, Col3 ~35%
        $col1X = $startX + 5;
        $col1W = 50;
        
        $col2X = $col1X + $col1W + 5;
        $col2W = 30;
        
        $col3X = $col2X + $col2W + 5;
        $col3W = 60;
        
        // --- COLUMNA 1: IMAGEN ---
        if (!empty($imgData) && file_exists($imgData)) {
            // Image($file, $x, $y, $w, $h, $type, $link, $align, $resize, $dpi, $palign, $ismask, $imgmask, $border, $fitbox, $hidden, $fitonpage)
            // Ajustar imagen para que ocupe casi todo el alto (dejando margen)
            $imgH = $boxHeight - 50; 
            $this->Image($imgData, $col1X-5, $currY + 5, $col1W+15, $imgH+15, '', '', 'T', false, 300, '', false, false, 0, 'CM'); 
        } else {
            // Placeholder si no hay imagen
            $this->SetXY($col1X, $currY + ($boxHeight/2) - 5);
            $this->SetFont('helvetica', 'I', 10);
            $this->Cell($col1W, 10, 'Sin Imagen', 0, 0, 'C');
        }
        
        // --- COLUMNA 2: A, B, C, D ---
        $this->SetXY($col2X, $currY + 5);
        $lineHeight = 8; // Espacio vertical entre elementos
        
        foreach ($col2Data as $key => $value) {
            $this->SetX($col2X);
            $this->SetFont('helvetica', 'B', 10);
            $this->Cell(10, 5, $key . ' ', 0, 0, 'L');
            $this->SetFont('helvetica', '', 10);
            $this->Cell(30, 5, $value.' m', 0, 1, 'L');
            
            $this->Ln(1); // Espacio extra
        }
        
        // --- COLUMNA 3: Etiquetas Mixtas ---
        $this->SetXY($col3X, $currY + 5);
        
        foreach ($col3Data as $item) {
            $this->SetX($col3X);
            
            // Determinar estilo según tipo (normal, pequeño, color)
            $fontSize = 10;
            $style = '';
            $color = [33, 37, 41];
            
            if (isset($item['size']) && $item['size'] == 'small') {
                $fontSize = 8;
                $this->SetTextColor(108, 117, 125); // Gris para pequeños
            } elseif (isset($item['color'])) {
                $color = $item['color'];
                $this->SetTextColorArray($color);
            } else {
                $this->SetTextColorArray($color);
            }
            
            if (isset($item['bold']) && $item['bold']) {
                $style = 'B';
            }

            if(isset($item['bolita']) && $item['bolita']) {
                // Usar el color del item si está definido
                if (isset($item['colorbolita']) && $item['colorbolita'] != '') {
                    $c = $item['colorbolita'];
                    $this->SetFillColorHex($c);
                } else {//Color de fondo del recuadro
                    $this->SetFillColor(213, 228, 231);
                    $this->SetDrawColor(213, 228, 231);
                }
                
                // Dibujar círculo relleno con borde y relleno ('DF') para asegurar que sea sólido
                // Radio = 1.5mm (diámetro 3mm)
                // Circle(x, y, r, ang_start, ang_end, style, line_style, fill_color, nc)
                $this->Circle($col3X + 2, $this->GetY() + 3, 1.5, 0, 360, 'DF');
                
                // Mover cursor a la derecha para el texto
                $this->SetX($col3X + 6);
                
                // Restaurar color de dibujo (opcional, pero buena práctica)
                //$this->SetDrawColor(0, 0, 0);
            } else {
                $this->SetX($col3X);
            }
            
            $this->SetFont('helvetica', $style, $fontSize);
            $this->Cell($col3W, 6, $item['text'], 0, 1, 'L');
        }
        
        // Actualizar posición final
        $this->SetY($currY + $boxHeight + 5);
    }

    public function setFillColorHex(string $hex): void
    {
        $hex = ltrim($hex, '#');

        // Soporta formato corto #F00
        if (strlen($hex) === 3) {
            $hex = $hex[0].$hex[0].$hex[1].$hex[1].$hex[2].$hex[2];
        }

        // Convierte a enteros RGB
        $r = hexdec(substr($hex, 0, 2));
        $g = hexdec(substr($hex, 2, 2));
        $b = hexdec(substr($hex, 4, 2));

        $this->SetFillColor($r, $g, $b);
        $this->SetDrawColor($r, $g, $b);
    }
}