<?php

namespace App\Http\Controllers;

use App\Models\Calle;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;
use Illuminate\Validation\ValidationException;
use Illuminate\Database\QueryException;
use Exception;

class RCalleController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        try {
            $query = Calle::with('codigoPostal');

            if ($request->has('id_codigo_postal')) {
                $query->where('id_codigo_postal', $request->id_codigo_postal);
            }

            $calles = $query->get();
            return response()->json(['success' => true, 'data' => $calles]);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Error al obtener las calles.', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            DB::beginTransaction();
            $data = $request->validate([
                'id_codigo_postal' => 'required|exists:codigo_postal,id',
                'nombre' => 'required|string|max:255|unique:calles,nombre',
                'estado' => 'required|string|max:1',
            ]);

            $calle = Calle::create($data);
            DB::commit();

            return response()->json(['success' => true, 'message' => 'Calle creada exitosamente.', 'data' => $calle], 201);
        } catch (ValidationException $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Errores de validación.', 'errors' => $e->errors()], 422);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Error al crear la calle.', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        try {
            $calle = Calle::with('codigoPostal')->findOrFail($id);
            return response()->json(['success' => true, 'data' => $calle]);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Calle no encontrada.'], 404);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        try {
            DB::beginTransaction();
            $calle = Calle::findOrFail($id);

            $data = $request->validate([
                'id_codigo_postal' => 'required|exists:codigo_postal,id',
                'nombre' => ['required', 'string', 'max:255', Rule::unique('calles', 'nombre')->ignore($calle->id)],
                'estado' => 'required|string|max:1',
            ]);

            $calle->update($data);
            DB::commit();

            return response()->json(['success' => true, 'message' => 'Calle actualizada exitosamente.', 'data' => $calle]);
        } catch (ValidationException $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Errores de validación.', 'errors' => $e->errors()], 422);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Error al actualizar la calle.', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        try {
            DB::beginTransaction();
            $calle = Calle::findOrFail($id);

            $calle->delete();
            DB::commit();

            return response()->json(['success' => true, 'message' => 'Calle eliminada exitosamente.']);
        } catch (Exception $e) {
            DB::rollBack();
            if ($e instanceof QueryException && $e->errorInfo[1] == 1451) {
                return response()->json(['success' => false, 'message' => 'No se puede eliminar la calle porque tiene registros asociados.'], 409);
            }
            return response()->json(['success' => false, 'message' => 'Error al eliminar la calle.', 'error' => $e->getMessage()], 500);
        }
    }
}
