<?php

namespace App\Http\Controllers;

use App\Models\Casa;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;
use Illuminate\Validation\ValidationException;
use Illuminate\Database\QueryException;
use Exception;

class RCasaController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        try {
            $query = Casa::with('calle');

            if ($request->has('id_calle')) {
                $query->where('id_calle', $request->id_calle);
            }

            $casas = $query->get();
            return response()->json(['success' => true, 'data' => $casas]);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Error al obtener las casas.', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            DB::beginTransaction();
            $data = $request->validate([
                'id_calle' => 'required|exists:calles,id',
                'numero' => 'required|string|max:20',
                'estado' => 'required|string|max:1',
            ]);

            $casa = Casa::create($data);
            DB::commit();

            return response()->json(['success' => true, 'message' => 'Casa creada exitosamente.', 'data' => $casa], 201);
        } catch (ValidationException $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Errores de validación.', 'errors' => $e->errors()], 422);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Error al crear la casa.', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        try {
            $casa = Casa::with('calle')->findOrFail($id);
            return response()->json(['success' => true, 'data' => $casa]);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Casa no encontrada.'], 404);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        try {
            DB::beginTransaction();
            $casa = Casa::findOrFail($id);

            $data = $request->validate([
                'id_calle' => 'required|exists:calles,id',
                'numero' => 'required|string|max:20',
                'estado' => 'required|string|max:1',
            ]);

            $casa->update($data);
            DB::commit();

            return response()->json(['success' => true, 'message' => 'Casa actualizada exitosamente.', 'data' => $casa]);
        } catch (ValidationException $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Errores de validación.', 'errors' => $e->errors()], 422);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Error al actualizar la casa.', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        try {
            DB::beginTransaction();
            $casa = Casa::findOrFail($id);

            $casa->delete();
            DB::commit();

            return response()->json(['success' => true, 'message' => 'Casa eliminada exitosamente.']);
        } catch (Exception $e) {
            DB::rollBack();
            if ($e instanceof QueryException && $e->errorInfo[1] == 1451) {
                return response()->json(['success' => false, 'message' => 'No se puede eliminar la casa porque tiene registros asociados.'], 409);
            }
            return response()->json(['success' => false, 'message' => 'Error al eliminar la casa.', 'error' => $e->getMessage()], 500);
        }
    }
}
