<?php

namespace App\Http\Controllers;

use App\Models\CodigoPostal;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;
use Illuminate\Validation\ValidationException;
use Illuminate\Database\QueryException;
use Exception;

class RCodigoPostalController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        try {
            $query = CodigoPostal::with('ubicacion');

            if ($request->has('id_ubicacion')) {
                $query->where('id_ubicacion', $request->id_ubicacion);
            }

            $codigosPostales = $query->get();
            return response()->json(['success' => true, 'data' => $codigosPostales]);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Error al obtener los códigos postales.', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            DB::beginTransaction();
            $data = $request->validate([
                'id_ubicacion' => 'required|exists:ubicacion,id',
                'nombre' => 'required|string|max:255|unique:codigo_postal,nombre',
                'estado' => 'required|string|max:1',
            ]);

            $codigoPostal = CodigoPostal::create($data);
            DB::commit();

            return response()->json(['success' => true, 'message' => 'Código Postal creado exitosamente.', 'data' => $codigoPostal], 201);
        } catch (ValidationException $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Errores de validación.', 'errors' => $e->errors()], 422);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Error al crear el código postal.', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        try {
            $codigoPostal = CodigoPostal::with('ubicacion')->findOrFail($id);
            return response()->json(['success' => true, 'data' => $codigoPostal]);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Código Postal no encontrado.'], 404);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        try {
            DB::beginTransaction();
            $codigoPostal = CodigoPostal::findOrFail($id);

            $data = $request->validate([
                'id_ubicacion' => 'required|exists:ubicacion,id',
                'nombre' => ['required', 'string', 'max:255', Rule::unique('codigo_postal', 'nombre')->ignore($codigoPostal->id)],
                'estado' => 'required|string|max:1',
            ]);

            $codigoPostal->update($data);
            DB::commit();

            return response()->json(['success' => true, 'message' => 'Código Postal actualizado exitosamente.', 'data' => $codigoPostal]);
        } catch (ValidationException $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Errores de validación.', 'errors' => $e->errors()], 422);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Error al actualizar el código postal.', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        try {
            DB::beginTransaction();
            $codigoPostal = CodigoPostal::findOrFail($id);

            $codigoPostal->delete();
            DB::commit();

            return response()->json(['success' => true, 'message' => 'Código Postal eliminado exitosamente.']);
        } catch (Exception $e) {
            DB::rollBack();
            if ($e instanceof QueryException && $e->errorInfo[1] == 1451) {
                return response()->json(['success' => false, 'message' => 'No se puede eliminar el código postal porque tiene registros asociados.'], 409);
            }
            return response()->json(['success' => false, 'message' => 'Error al eliminar el código postal.', 'error' => $e->getMessage()], 500);
        }
    }
}
