<?php

namespace App\Http\Controllers;

use App\Models\HpImagen;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;
use Illuminate\Database\QueryException;
use Exception;

class RHpImagenController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        try {
            $query = HpImagen::with('instalacion');

            if ($request->has('id_hp_instalacion')) {
                $query->where('id_hp_instalacion', $request->id_hp_instalacion);
            }

            $imagenes = $query->get();
            return response()->json(['success' => true, 'data' => $imagenes]);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Error al obtener las imágenes.', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            DB::beginTransaction();
            $data = $request->validate([
                'id_hp_instalacion' => 'required|exists:hp_instalacion,id',
                'nombre_imagen' => 'required|string|max:255',
                'descripcion' => 'nullable|string',
                'estado' => 'required|string|max:1',
            ]);

            $imagen = HpImagen::create($data);
            DB::commit();

            return response()->json(['success' => true, 'message' => 'Imagen creada exitosamente.', 'data' => $imagen], 201);
        } catch (ValidationException $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Errores de validación.', 'errors' => $e->errors()], 422);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Error al crear la imagen.', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Upload an image file
     */
    public function upload(Request $request)
    {
        try {
            DB::beginTransaction();
            
            $request->validate([
                'id_hp_instalacion' => 'required|exists:hp_instalacion,id',
                'imagen' => 'required|image|mimes:jpeg,png,jpg,gif|max:5120', // Max 5MB
                'descripcion' => 'nullable|string',
            ]);

            $instalacionId = $request->id_hp_instalacion;
            
            // Handle file upload
            if ($request->hasFile('imagen')) {
                $file = $request->file('imagen');
                
                // Generate unique filename
                $filename = time() . '_' . uniqid() . '.' . $file->getClientOriginalExtension();
                
                // Define storage path
                $destinationPath = public_path('assets/images/jobs');
                
                // Create directory if it doesn't exist
                if (!file_exists($destinationPath)) {
                    mkdir($destinationPath, 0755, true);
                }
                
                // Move file to destination
                $file->move($destinationPath, $filename);
                
                // Create database record
                $imagen = HpImagen::create([
                    'id_hp_instalacion' => $instalacionId,
                    'nombre_imagen' => $filename,
                    'descripcion' => $request->descripcion,
                    'estado' => 'S'
                ]);
                
                DB::commit();
                
                return response()->json([
                    'success' => true, 
                    'message' => 'Imagen subida exitosamente.', 
                    'data' => $imagen
                ], 201);
            }
            
            return response()->json([
                'success' => false, 
                'message' => 'No se recibió ningún archivo.'
            ], 400);
            
        } catch (ValidationException $e) {
            DB::rollBack();
            return response()->json([
                'success' => false, 
                'message' => 'Errores de validación.', 
                'errors' => $e->errors()
            ], 422);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false, 
                'message' => 'Error al subir la imagen.', 
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        try {
            $imagen = HpImagen::with('instalacion')->findOrFail($id);
            return response()->json(['success' => true, 'data' => $imagen]);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Imagen no encontrada.'], 404);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        try {
            DB::beginTransaction();
            $imagen = HpImagen::findOrFail($id);

            $data = $request->validate([
                'id_hp_instalacion' => 'required|exists:hp_instalacion,id',
                'nombre_imagen' => 'required|string|max:255',
                'descripcion' => 'nullable|string',
                'estado' => 'required|string|max:1',
            ]);

            $imagen->update($data);
            DB::commit();

            return response()->json(['success' => true, 'message' => 'Imagen actualizada exitosamente.', 'data' => $imagen]);
        } catch (ValidationException $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Errores de validación.', 'errors' => $e->errors()], 422);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Error al actualizar la imagen.', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        try {
            DB::beginTransaction();
            $imagen = HpImagen::findOrFail($id);

            $imagen->delete();
            DB::commit();

            return response()->json(['success' => true, 'message' => 'Imagen eliminada exitosamente.']);
        } catch (Exception $e) {
            DB::rollBack();
            if ($e instanceof QueryException && $e->errorInfo[1] == 1451) {
                return response()->json(['success' => false, 'message' => 'No se puede eliminar la imagen porque tiene registros asociados.'], 409);
            }
            return response()->json(['success' => false, 'message' => 'Error al eliminar la imagen.', 'error' => $e->getMessage()], 500);
        }
    }
}
