<?php

namespace App\Http\Controllers;

use App\Models\HpInstalacion;
use App\Models\VListaAsignacion;
use App\Models\VAsignacionEquipo;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;
use Illuminate\Database\QueryException;
use Yajra\DataTables\Facades\DataTables;
use Carbon\Carbon;
use Exception;

class RHpInstalacionController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        try {
            $query = HpInstalacion::with(['casa', 'persona']);

            if ($request->has('id_casa')) {
                $query->where('id_casa', $request->id_casa);
            }

            if ($request->has('id_persona')) {
                $query->where('id_persona', $request->id_persona);
            }

            $instalaciones = $query->get();
            return response()->json(['success' => true, 'data' => $instalaciones]);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Error al obtener las instalaciones.', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            DB::beginTransaction();
            $data = $request->validate([
                'id_casa' => 'required|exists:casas,id',
                'id_persona' => 'required|exists:personas,id',
                'id_persona_supervisor' => 'required|numeric',
                'tipo_objeto' => 'nullable|string|max:255',
                'numero_unidades_residenciales' => 'nullable|integer',
                'vivienda_unifamiliar' => 'nullable|boolean',
                'vivienda_multifamiliar' => 'nullable|boolean',
                'A_' => 'nullable|numeric',
                'B_' => 'nullable|numeric',
                'Do_' => 'nullable|numeric',
                'D_' => 'nullable|numeric',
                'latitude' => 'nullable|string|max:255',
                'longitude' => 'nullable|string|max:255',
                'exactitud' => 'nullable|string|max:255',
                'deteccion_manual' => 'nullable|boolean',
                'asociacion' => 'nullable|string|max:255',
                'microducto' => 'nullable|string|max:255',
            ]);
            $data['id_estado_instalacion'] = '1'; //Asignado
            $data['fecha_asignacion'] = now()->format('Y-m-d H:i:s');
            $data['id_persona_asignacion'] = auth()->user()->id_persona;
            $data['estado'] = 'S';

            $instalacion = HpInstalacion::create($data);
            DB::commit();

            return response()->json(['success' => true, 'message' => 'Instalación creada exitosamente.', 'data' => $instalacion], 201);
        } catch (ValidationException $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Errores de validación.', 'errors' => $e->errors()], 422);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Error al crear la instalación.', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        try {
            $instalacion = HpInstalacion::with(['casa', 'persona'])->findOrFail($id);
            return response()->json(['success' => true, 'data' => $instalacion]);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Instalación no encontrada.'], 404);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        try {
            DB::beginTransaction();
            $instalacion = HpInstalacion::findOrFail($id);
            //El equipo y supervisor ya estan asignados no deben reasignarse

            $data = $request->validate([
                'id_casa' => 'required|exists:casas,id',
                'id_persona' => 'nullable|exists:personas,id',
                'id_tipo_objeto' => 'nullable|exists:tipo_objeto,id',
                'numero_unidades_residenciales' => 'nullable|integer',
                'vivienda_unifamiliar' => 'nullable|boolean',
                'vivienda_multifamiliar' => 'nullable|boolean',
                'A_' => 'nullable|numeric',
                'B_' => 'nullable|numeric',
                'Do_' => 'nullable|numeric',
                'D_' => 'nullable|numeric',
                'latitude' => 'nullable|string|max:255',
                'longitude' => 'nullable|string|max:255',
                'exactitud' => 'nullable|string|max:255',
                'deteccion_manual' => 'nullable|boolean',
                'id_color_asociacion' => 'nullable|exists:colores,id',
                'id_color_microducto' => 'nullable|exists:colores,id',
                'id_estado_instalacion' => 'nullable|exists:estado_instalacion,id',
            ]);
            $fecha = now()->format('Y-m-d H:i:s');
            $data['fecha_asignacion'] = $fecha;

            // Update dates based on status
            if(isset($data['id_estado_instalacion'])) {
                if($data['id_estado_instalacion'] == '1') {
                    $data['fecha_inicio'] = $fecha;
                }
                if($data['id_estado_instalacion'] == '2') {
                    $data['fecha_finalizacion'] = $fecha;
                }
                if($data['id_estado_instalacion'] == '3') {
                    $data['fecha_cancelacion'] = $fecha;
                }
            }
            $data['estado'] = 'S';

            $instalacion->update($data);
            DB::commit();

            return response()->json(['success' => true, 'message' => 'Instalación actualizada exitosamente.', 'data' => $instalacion]);
        } catch (ValidationException $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Errores de validación.', 'errors' => $e->errors()], 422);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Error al actualizar la instalación.', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        try {
            DB::beginTransaction();
            $instalacion = HpInstalacion::findOrFail($id);

            $instalacion->delete();
            DB::commit();

            return response()->json(['success' => true, 'message' => 'Instalación eliminada exitosamente.']);
        } catch (Exception $e) {
            DB::rollBack();
            if ($e instanceof QueryException && $e->errorInfo[1] == 1451) {
                return response()->json(['success' => false, 'message' => 'No se puede eliminar la instalación porque tiene registros asociados.'], 409);
            }
            return response()->json(['success' => false, 'message' => 'Error al eliminar la instalación.', 'error' => $e->getMessage()], 500);
        }
    }
    
    public function getAllInstalaciones()
    {
        
    }

    public  function getById(string $id)
    {
        try {
            $instalacion = HpInstalacion::where('id', $id)->first();
            return response()->json(['success' => true, 'data' => $instalacion]);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Instalación no encontrada.'], 404);
        }
    }

    public  function getInstalacionById(string $id)
    {
        try {
            $instalacion = VAsignacionEquipo::where('id', $id)->first();
            return response()->json(['success' => true, 'data' => $instalacion]);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Instalación no encontrada.'], 404);
        }
    }

    public function getData()
    {
        try {
            $query = VListaAsignacion::query();
            
            return DataTables::of($query)
                ->addColumn('nombre_completo', function ($row) {
                    return $row->paterno . ' ' . $row->materno . ', ' . $row->nombre_persona;
                })
                ->addColumn('direccion', function ($row) {
                    return $row->nombre_ubicacion . ' CP:' . $row->nombre_codigo_postal . ' Calle:' . $row->nombre_calle. ' #' . $row->numero;
                })
                ->addColumn('estado_instalacion', function ($row) {
                    return '<span class="px-2 py-1 text-xs font-medium rounded-full bg-'.$row->nombre_color.'-100 text-'.$row->nombre_color.'-800">'.$row->nombre_estado_instalacion.'</span>';
                })
                ->addColumn('fecha_asignacion', function ($row) {
                    return Carbon::parse($row->fecha_asignacion)->format('Y-m-d');
                })
                ->addColumn('action', function($row) {

                    $btnVer = '<button 
                                    class="inline-flex items-center gap-1 px-3 py-1 bg-blue-50 text-blue-700 font-semibold rounded-md 
                                        hover:bg-blue-200 transition-colors mr-1 btn-ver"
                                    data-id="'.$row->id.'">
                                        <i class="material-symbols-outlined text-sm" 
                                        style="font-variation-settings: \'wght\' 200, \'GRAD\' 0;">
                                            visibility
                                        </i>
                                        Ver
                                </button>';

                    $btnEditar = '<a href="'.url('/dashboard/hpinstalacionjob/'.$row->id.'/edit').'" 
                                class="inline-flex items-center gap-1 px-3 py-1 bg-amber-50 text-amber-700 font-semibold rounded-md
                                        hover:bg-amber-200 transition-colors mr-1">
                                        <i class="material-symbols-outlined text-sm" 
                                        style="font-variation-settings: \'wght\' 200, \'GRAD\' 0;">
                                            edit
                                        </i>
                                        Editar
                                </a>';

                    $btnPdf = '<a href="'.url('/dashboard/informejob/'.$row->id).'" 
                                target="_blank" rel="noopener noreferrer"
                                class="inline-flex items-center gap-1 px-3 py-1 bg-red-50 text-red-700 font-semibold rounded-md
                                        hover:bg-red-200 transition-colors">
                                        <i class="material-symbols-outlined text-sm" 
                                        style="font-variation-settings: \'wght\' 200, \'GRAD\' 0;">
                                            picture_as_pdf
                                        </i>
                                        PDF
                                </a>';

                    //return '<div class="text-center">' . $btnVer . $btnEditar . $btnPdf . '</div>';
                    return '<div class="text-center">' . $btnEditar . $btnPdf . '</div>';
                })
                ->rawColumns(['action', 'estado_instalacion'])
                ->make(true);
        } catch (Exception $e) {
            \Log::error('Error en getData de RHpInstalacionController: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());
            
            return response()->json([
                'draw' => 0,
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => 'Error al cargar los datos: ' . $e->getMessage()
            ], 500);
        }
    }

    public function getByDate(Request $request)
    {
        try {
            $fecha = $request->input('fecha', now()->format('Y-m-d'));

            $instalaciones = VListaAsignacion::query()
                ->whereDate('fecha_asignacion', $fecha)
                ->orderBy('fecha_asignacion', 'desc')
                ->get();
            
            
             return DataTables::of($instalaciones)
                ->addColumn('direccion', function ($row) {
                     return $row->nombre_ubicacion . ' CP:' . $row->nombre_codigo_postal . ' Calle:' . $row->nombre_calle. ' #' . $row->numero;
                })
                ->addColumn('equipo', function ($row) {
                    return $row->paterno . ' ' . $row->materno . ', ' . $row->nombre_persona;
                })
                ->addColumn('supervisor', function ($row) {
                    return '';
                })
                ->addColumn('estado_instalacion', function ($row) { //--- Esto falta corregir
                    return '<span class="px-2 py-1 text-xs font-medium rounded-full bg-'.$row->nombre_color.'-100 text-'.$row->nombre_color.'-800">'.$row->nombre_estado_instalacion.'</span>';
                })
                ->addColumn('fecha_asignacion', function ($row) {
                    return Carbon::parse($row->fecha_asignacion)->format('Y-m-d');
                    //return $row->fecha_asignacion ? $row->fecha_asignacion->format('d/m/Y') : '';
                })
                ->addColumn('action', function($row) {
                    // Botones Ver y Eliminar
                    $btnVer = '<button type="button" class="btn-ver bg-blue-50 hover:bg-blue-100 text-blue-500 p-1.5 rounded-lg mr-1 transition-colors" data-id="'.$row->id.'" title="Ver detalles">
                                    <span class="material-symbols-outlined text-xl">visibility</span>
                               </button>';
                    
                    $btnEliminar = '<button type="button" class="btn-eliminar bg-red-50 hover:bg-red-100 text-red-500 p-1.5 rounded-lg transition-colors" data-id="'.$row->id.'" title="Eliminar asignación">
                                        <span class="material-symbols-outlined text-xl">delete</span>
                                    </button>';

                    return '<div class="flex items-center justify-center">'.$btnVer.$btnEliminar.'</div>';
                })
                ->rawColumns(['action', 'estado_instalacion'])
                ->make(true);

        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Error al obtener asignaciones por fecha.', 'error' => $e->getMessage()], 500);
        }
    }
}
