<?php

namespace App\Http\Controllers;

use App\Models\VPersonaUsuario;
use App\Models\Persona;
use App\Helpers\RolesEnum;
use Illuminate\Http\Request;
use Yajra\DataTables\Facades\DataTables;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;
use Illuminate\Validation\ValidationException;
use Exception;

class RPersonaController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $personas = VPersonaUsuario::query()->select([
            'id',
            'dni',
            'paterno',
            'materno',
            'nombre',
            'email',
            'telefono_celular',
            'estado_usuario',
            'id_usuario'
        ])->where('id_rol', RolesEnum::TECNICO);

        return DataTables::of($personas)
            ->addColumn('nombre_completo', function ($persona) {
                return $persona->paterno . ' ' . $persona->materno . ', ' . $persona->nombre;
            })
            ->addColumn('action', function ($persona) {
                $viewBtn = '<button data-id="' . $persona->id . '" class="btn-ver bg-indigo-50 hover:bg-indigo-100 text-indigo-400 px-2 py-1 rounded-md mr-1 inline-flex items-center">' .
                    '<span class="material-symbols-outlined text-base align-middle">visibility</span>' .
                    '</button>';

                $editBtn = '<button data-id="' . $persona->id . '" class="btn-editar bg-green-50 hover:bg-green-100 text-green-400 px-2 py-1 rounded-md mr-1 inline-flex items-center">' .
                    '<span class="material-symbols-outlined text-base align-middle">edit</span>' .
                    '</button>';

                $deleteBtn = '<button data-id="' . $persona->id . '" class="btn-eliminar bg-red-50 hover:bg-red-100 text-red-400 px-2 py-1 rounded-md inline-flex items-center">' .
                    '<span class="material-symbols-outlined text-base align-middle">delete</span>' .
                    '</button>';
                
                $resetBtn = '<a href="' . url('/dashboard/usuario/'.$persona->id.'/reset') . '" 
                        class="bg-gray-100 hover:bg-gray-200 text-gray-700 px-2 py-1 rounded-md inline-flex items-center ml-2">
                            <span class="material-symbols-outlined text-base align-middle">lock_reset</span>
                        </a>';

                return $viewBtn . $editBtn . $deleteBtn . $resetBtn;
            })
            ->rawColumns(['action'])
            ->make(true);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            DB::beginTransaction();
            $data = $request->validate([
                'id_proyecto' => 'required|exists:proyectos,id',
                'dni' => 'required|string|max:20|unique:personas,dni',
                'paterno' => 'required|string|max:100',
                'materno' => 'nullable|string|max:100',
                'nombre' => 'required|string|max:100',
                'nombre_corto' => 'nullable|string|max:50',
                'email' => 'nullable|email|max:255',
                'telefono_celular' => 'nullable|string|max:20',
                'telefono_fijo' => 'nullable|string|max:20',
                'direccion' => 'nullable|string|max:255',
                'fecha_nacimiento' => 'nullable|date',
                'zona' => 'nullable|string|max:100',
                'Website' => 'nullable|string|max:255',
                'Twitter' => 'nullable|string|max:255',
                'fotografia' => 'nullable|string|max:255',
            ]);

            // Valores por defecto
            $data['estado'] = 'S';

            $persona = Persona::create($data);
            DB::commit();

            return response()->json(['success' => true, 'message' => 'Persona creada exitosamente.', 'data' => $persona], 201);
        } catch (ValidationException $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Errores de validación.', 'errors' => $e->errors()], 422);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Error al crear la persona.', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        try {
            $persona = Persona::findOrFail($id);
            return response()->json(['success' => true, 'data' => $persona]);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Persona no encontrada.'], 404);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        try {
            DB::beginTransaction();
            $persona = Persona::findOrFail($id);

            $data = $request->validate([
                'id_proyecto' => 'required|exists:proyectos,id',
                'dni' => ['required', 'string', 'max:20', Rule::unique('personas', 'dni')->ignore($persona->id)],
                'paterno' => 'required|string|max:100',
                'materno' => 'nullable|string|max:100',
                'nombre' => 'required|string|max:100',
                'nombre_corto' => 'nullable|string|max:50',
                'email' => 'nullable|email|max:255',
                'telefono_celular' => 'nullable|string|max:20',
                'telefono_fijo' => 'nullable|string|max:20',
                'direccion' => 'nullable|string|max:255',
                'fecha_nacimiento' => 'nullable|date',
                'zona' => 'nullable|string|max:100',
                'Website' => 'nullable|string|max:255',
                'Twitter' => 'nullable|string|max:255',
                'fotografia' => 'nullable|string|max:255',
                'estado' => 'nullable|string|max:1',
            ]);

            $persona->update($data);
            DB::commit();

            return response()->json(['success' => true, 'message' => 'Persona actualizada exitosamente.', 'data' => $persona]);
        } catch (ValidationException $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Errores de validación.', 'errors' => $e->errors()], 422);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Error al actualizar la persona.', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        try {
            DB::beginTransaction();
            $persona = Persona::findOrFail($id);

            $persona->delete();
            DB::commit();

            return response()->json(['success' => true, 'message' => 'Persona eliminada exitosamente.']);
        } catch (Exception $e) {
            DB::rollBack();
            if ($e instanceof \Illuminate\Database\QueryException && $e->errorInfo[1] == 1451) {
                return response()->json(['success' => false, 'message' => 'No se puede eliminar la persona porque tiene registros asociados.'], 409);
            }
            return response()->json(['success' => false, 'message' => 'Error al eliminar la persona.', 'error' => $e->getMessage()], 500);
        }
    }
    /**
     * Update the profile specific fields.
     */
    public function updateProfile(Request $request)
    {
        try {
            DB::beginTransaction();
            $user = auth()->user();
            $persona = $user->persona;

            if (!$persona) {
                return response()->json(['success' => false, 'message' => 'Persona no encontrada para el usuario actual.'], 404);
            }

            $data = $request->validate([
                'direccion' => 'nullable|string|max:255',
                'telefono_celular' => 'nullable|string|max:20',
                'fotografia_file' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048', // Validar archivo de imagen
            ]);

            if ($request->hasFile('fotografia_file')) {
                // Eliminar foto anterior si no es default
                if ($persona->fotografia && $persona->fotografia !== 'default.jpg') {
                    $oldPath = public_path('assets/images/users/' . $persona->fotografia);
                    if (file_exists($oldPath)) {
                        unlink($oldPath);
                    }
                }

                $file = $request->file('fotografia_file');
                $filename = time() . '_' . $file->getClientOriginalName();
                $file->move(public_path('assets/images/users'), $filename);
                $persona->fotografia = $filename;
            }

            if (isset($data['direccion'])) {
                $persona->direccion = $data['direccion'];
            }
            if (isset($data['telefono_celular'])) {
                $persona->telefono_celular = $data['telefono_celular'];
            }

            $persona->save();
            DB::commit();

            return response()->json(['success' => true, 'message' => 'Perfil actualizado correctamente.', 'data' => $persona]);

        } catch (ValidationException $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Errores de validación.', 'errors' => $e->errors()], 422);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Error al actualizar el perfil.', 'error' => $e->getMessage()], 500);
        }
    }
}

