<?php

namespace App\Http\Controllers;

use App\Models\Proyecto;
use Illuminate\Http\Request;
use Yajra\DataTables\Facades\DataTables;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;
use Illuminate\Validation\ValidationException;
use Exception;

class RProyectoController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $proyectos = Proyecto::query()->select([
            'id',
            'nombre',
            'direccion',
            'telefono',
            'email',
            'website',
            'estado'
        ]);

        return DataTables::of($proyectos)
            ->addColumn('action', function ($proyecto) {
                $viewBtn = '<button data-id="' . $proyecto->id . '" class="btn-ver bg-indigo-50 hover:bg-indigo-100 text-indigo-400 px-2 py-1 rounded-md mr-1 inline-flex items-center">' .
                    '<span class="material-symbols-outlined text-base align-middle">visibility</span>' .
                    '</button>';

                $editBtn = '<button data-id="' . $proyecto->id . '" class="btn-editar bg-green-50 hover:bg-green-100 text-green-400 px-2 py-1 rounded-md mr-1 inline-flex items-center">' .
                    '<span class="material-symbols-outlined text-base align-middle">edit</span>' .
                    '</button>';

                $deleteBtn = '<button data-id="' . $proyecto->id . '" class="btn-eliminar bg-red-50 hover:bg-red-100 text-red-400 px-2 py-1 rounded-md inline-flex items-center">' .
                    '<span class="material-symbols-outlined text-base align-middle">delete</span>' .
                    '</button>';

                return $viewBtn . $editBtn . $deleteBtn;
            })
            ->rawColumns(['action'])
            ->make(true);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            DB::beginTransaction();
            $data = $request->validate([
                'nombre' => 'required|string|max:255|unique:proyectos,nombre',
                'direccion' => 'nullable|string|max:255',
                'telefono' => 'nullable|string|max:50',
                'email' => 'nullable|email|max:255',
                'website' => 'nullable|string|max:255',
                'logo_compania' => 'nullable|string|max:255',
            ]);

            // Valores por defecto
            $data['estado'] = 'S'; 

            $proyecto = Proyecto::create($data);
            DB::commit();

            return response()->json(['success' => true, 'message' => 'Proyecto creado exitosamente.', 'data' => $proyecto], 201);
        } catch (ValidationException $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Errores de validación.', 'errors' => $e->errors()], 422);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Error al crear el proyecto.', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        try {
            $proyecto = Proyecto::findOrFail($id);
            return response()->json(['success' => true, 'data' => $proyecto]);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Proyecto no encontrado.'], 404);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        try {
            DB::beginTransaction();
            $proyecto = Proyecto::findOrFail($id);

            $data = $request->validate([
                'nombre' => ['required', 'string', 'max:255', Rule::unique('proyectos', 'nombre')->ignore($proyecto->id)],
                'direccion' => 'nullable|string|max:255',
                'telefono' => 'nullable|string|max:50',
                'email' => 'nullable|email|max:255',
                'website' => 'nullable|string|max:255',
                'logo_compania' => 'nullable|string|max:255',
                'estado' => 'nullable|string|max:1',
            ]);

            $proyecto->update($data);
            DB::commit();

            return response()->json(['success' => true, 'message' => 'Proyecto actualizado exitosamente.', 'data' => $proyecto]);
        } catch (ValidationException $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Errores de validación.', 'errors' => $e->errors()], 422);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Error al actualizar el proyecto.', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        try {
            DB::beginTransaction();
            $proyecto = Proyecto::findOrFail($id);

            $proyecto->delete();
            DB::commit();

            return response()->json(['success' => true, 'message' => 'Proyecto eliminado exitosamente.']);
        } catch (Exception $e) {
            DB::rollBack();
            if ($e instanceof \Illuminate\Database\QueryException && $e->errorInfo[1] == 1451) {
                return response()->json(['success' => false, 'message' => 'No se puede eliminar el proyecto porque tiene registros asociados.'], 409);
            }
            return response()->json(['success' => false, 'message' => 'Error al eliminar el proyecto.', 'error' => $e->getMessage()], 500);
        }
    }
}
