<?php

namespace App\Http\Controllers;

use App\Models\Ubicacion;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;
use Illuminate\Validation\ValidationException;
use Illuminate\Database\QueryException;
use Exception;

class RUbicacionController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        try {
            $ubicaciones = Ubicacion::all();
            return response()->json(['success' => true, 'data' => $ubicaciones]);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Error al obtener las ubicaciones.', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            DB::beginTransaction();
            $data = $request->validate([
                'nombre' => 'required|string|max:255|unique:ubicacion,nombre',
                'estado' => 'required|string|max:1',
            ]);

            $ubicacion = Ubicacion::create($data);
            DB::commit();

            return response()->json(['success' => true, 'message' => 'Ubicación creada exitosamente.', 'data' => $ubicacion], 201);
        } catch (ValidationException $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Errores de validación.', 'errors' => $e->errors()], 422);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Error al crear la ubicación.', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        try {
            $ubicacion = Ubicacion::findOrFail($id);
            return response()->json(['success' => true, 'data' => $ubicacion]);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Ubicación no encontrada.'], 404);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        try {
            DB::beginTransaction();
            $ubicacion = Ubicacion::findOrFail($id);

            $data = $request->validate([
                'nombre' => ['required', 'string', 'max:255', Rule::unique('ubicacion', 'nombre')->ignore($ubicacion->id)],
                'estado' => 'required|string|max:1',
            ]);

            $ubicacion->update($data);
            DB::commit();

            return response()->json(['success' => true, 'message' => 'Ubicación actualizada exitosamente.', 'data' => $ubicacion]);
        } catch (ValidationException $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Errores de validación.', 'errors' => $e->errors()], 422);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Error al actualizar la ubicación.', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        try {
            DB::beginTransaction();
            $ubicacion = Ubicacion::findOrFail($id);

            $ubicacion->delete();
            DB::commit();

            return response()->json(['success' => true, 'message' => 'Ubicación eliminada exitosamente.']);
        } catch (Exception $e) {
            DB::rollBack();
            if ($e instanceof QueryException && $e->errorInfo[1] == 1451) {
                return response()->json(['success' => false, 'message' => 'No se puede eliminar la ubicación porque tiene registros asociados.'], 409);
            }
            return response()->json(['success' => false, 'message' => 'Error al eliminar la ubicación.', 'error' => $e->getMessage()], 500);
        }
    }
}
