<?php

namespace App\Http\Controllers;

use App\Models\Usuario;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;
use Illuminate\Validation\ValidationException;
use Exception;
use App\Services\MailService;

class RUsuarioController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        try {
            $usuarios = Usuario::with(['persona', 'rol'])->get();
            return response()->json(['success' => true, 'data' => $usuarios]);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Error al obtener usuarios.', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request, MailService $mailService)
    {
        try {
            DB::beginTransaction();
            $data = $request->validate([
                'id_persona' => 'required|exists:personas,id|unique:usuarios,id_persona',
                'id_rol' => 'required|exists:roles,id',
                'email' => 'required|email|max:255|unique:usuarios,email',
                'password' => 'required|string|min:8',
                'clave_inicial' => 'required|string|min:8',
            ]);
            $data['username'] = $data['email'];
            $data['estado'] = 'S';
            $data['debe_cambiar_password'] = true;

            $usuario = Usuario::create($data);

            $mailService->sendWelcome(
                $data['email'],
                $data['clave_inicial']
            );

            DB::commit();

            return response()->json(['success' => true, 'message' => 'Usuario creado exitosamente.', 'data' => $usuario], 201);
        } catch (ValidationException $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Errores de validación.', 'errors' => $e->errors()], 422);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Error al crear el usuario.', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        try {
            $usuario = Usuario::with(['persona', 'rol'])->findOrFail($id);
            return response()->json(['success' => true, 'data' => $usuario]);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Usuario no encontrado.'], 404);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        try {
            DB::beginTransaction();
            $usuario = Usuario::findOrFail($id);

            $data = $request->validate([
                'id_persona' => ['required', 'exists:personas,id', Rule::unique('usuarios', 'id_persona')->ignore($usuario->id)],
                'id_rol' => 'required|exists:roles,id',
                'email' => ['required', 'email', 'max:255', Rule::unique('usuarios', 'email')->ignore($usuario->id)],
                'password' => 'nullable|string|min:8',
                'estado' => 'nullable|string|max:1',
            ]);

            $usuario->update($data);
            DB::commit();

            return response()->json(['success' => true, 'message' => 'Usuario actualizado exitosamente.', 'data' => $usuario]);
        } catch (ValidationException $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Errores de validación.', 'errors' => $e->errors()], 422);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Error al actualizar el usuario.', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        try {
            DB::beginTransaction();
            $usuario = Usuario::findOrFail($id);

            $usuario->delete();
            DB::commit();

            return response()->json(['success' => true, 'message' => 'Usuario eliminado exitosamente.']);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Error al eliminar el usuario.', 'error' => $e->getMessage()], 500);
        }
    }

    public function updatePassword(Request $request, $id)
    {
        try {
            DB::beginTransaction();
            $usuario = Usuario::findOrFail($id);
            
            $data = $request->validate([
                'password' => 'required|string|min:8'
            ]);

            $usuario->password = Hash::make($data['password']);
            $usuario->clave_inicial = $data['password'];
            $usuario->save();

            DB::commit();

            return response()->json(['success' => true, 'message' => 'Contraseña actualizada exitosamente.']);

        } catch (ValidationException $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Errores de validación.', 'errors' => $e->errors()], 422);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Error al actualizar la contraseña.', 'error' => $e->getMessage()], 500);
        }
    }
    public function updateProfileEmail(Request $request)
    {
        try {
            DB::beginTransaction();
            $usuario = auth()->user();

            $data = $request->validate([
                'email' => ['required', 'email', 'max:255', Rule::unique('usuarios', 'email')->ignore($usuario->id)],
            ]);

            $usuario->email = $data['email'];
            $usuario->username = $data['email']; // Actualizamos también el username si es igual al email
            $usuario->save();

            DB::commit();

            return response()->json(['success' => true, 'message' => 'Email actualizado correctamente.']);

        } catch (ValidationException $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Errores de validación.', 'errors' => $e->errors()], 422);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Error al actualizar el email.', 'error' => $e->getMessage()], 500);
        }
    }

    public function updateProfilePassword(Request $request)
    {
        try {
            DB::beginTransaction();
            $usuario = auth()->user();

            $data = $request->validate([
                'current_password' => 'required',
                'new_password' => 'required|string|min:8|confirmed',
            ]);

            if (!Hash::check($data['current_password'], $usuario->password)) {
                throw ValidationException::withMessages([
                    'current_password' => ['La contraseña actual es incorrecta.'],
                ]);
            }

            $usuario->password = Hash::make($data['new_password']);
            $usuario->save();

            DB::commit();

            return response()->json(['success' => true, 'message' => 'Contraseña actualizada correctamente.']);

        } catch (ValidationException $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Errores de validación.', 'errors' => $e->errors()], 422);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Error al actualizar la contraseña.', 'error' => $e->getMessage()], 500);
        }
    }
}

