/**
 * CRUD.js
 * 
 * A generic utility for handling CRUD operations via AJAX.
 * Dependencies: jQuery
 */

const CRUD = {
    /**
     * Initialize CSRF token for all AJAX requests.
     * Should be called once, or the setup runs automatically if this file is included after jQuery.
     */
    init: function() {
        $.ajaxSetup({
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            }
        });
    },

    /**
     * Fetch all records or a specific set based on params.
     * @param {string} url - The URL to fetch from.
     * @param {object} params - Optional query parameters.
     * @returns {Promise} - The jQuery XHR promise.
     */
    getAll: function(url, params = {}) {
        return $.ajax({
            url: url,
            type: 'GET',
            data: params
        });
    },

    /**
     * Fetch a single record by ID.
     * @param {string} url - The base URL (e.g., '/api/items').
     * @param {string|number} id - The ID of the record.
     * @returns {Promise} - The jQuery XHR promise.
     */
    getById: function(url, id) {
        // Ensure URL doesn't end with slash to avoid double slashes if user provides one
        const cleanUrl = url.replace(/\/$/, "");
        return $.ajax({
            url: `${cleanUrl}/${id}`,
            type: 'GET'
        });
    },

    /**
     * Create a new record.
     * @param {string} url - The URL to post to.
     * @param {object|FormData} data - The data to send.
     * @returns {Promise} - The jQuery XHR promise.
     */
    create: function(url, data) {
        const isFormData = data instanceof FormData;
        return $.ajax({
            url: url,
            type: 'POST',
            data: data,
            processData: !isFormData,
            contentType: isFormData ? false : 'application/x-www-form-urlencoded; charset=UTF-8'
        });
    },

    /**
     * Update an existing record.
     * @param {string} url - The base URL (e.g., '/api/items').
     * @param {string|number} id - The ID of the record.
     * @param {object|FormData} data - The data to send.
     * @returns {Promise} - The jQuery XHR promise.
     */
    update: function(url, id, data) {
        const cleanUrl = url.replace(/\/$/, "");
        const isFormData = data instanceof FormData;
        
        // If using FormData with PUT/PATCH, Laravel often requires _method: 'PUT' in the body
        // and the actual request to be POST, or using x-www-form-urlencoded.
        // For simplicity here, we'll assume standard PUT if not FormData, 
        // or let the user handle the _method spoofing if they pass FormData.
        
        return $.ajax({
            url: `${cleanUrl}/${id}`,
            type: 'PUT', // Default to PUT. If using FormData, caller might need to use POST with _method field.
            data: data,
            processData: !isFormData,
            contentType: isFormData ? false : 'application/x-www-form-urlencoded; charset=UTF-8'
        });
    },

    /**
     * Delete a record.
     * @param {string} url - The base URL.
     * @param {string|number} id - The ID of the record.
     * @returns {Promise} - The jQuery XHR promise.
     */
    delete: function(url, id) {
        const cleanUrl = url.replace(/\/$/, "");
        return $.ajax({
            url: `${cleanUrl}/${id}`,
            type: 'DELETE'
        });
    }
};

// Auto-initialize if jQuery is present
if (typeof $ !== 'undefined') {
    CRUD.init();
}
