@extends('master')

@section('stylesCSS')
    <link rel="stylesheet" href="https://cdn.datatables.net/2.3.5/css/dataTables.tailwindcss.css">
@endsection

@section('headerContent')
<h5 class="!mb-0">
    Gestión de Proyectos
</h5>
<ol class="breadcrumb mt-[12px] md:mt-0">
    <li class="breadcrumb-item inline-block relative text-sm mx-[11px]">
        <a href="/main" class="inline-block relative ltr:pl-[22px] rtl:pr-[22px] transition-all hover:text-primary-500">
            <i class="material-symbols-outlined absolute ltr:left-0 rtl:right-0 !text-lg -mt-px text-primary-500 top-1/2 -translate-y-1/2">home</i>
            Dashboard
        </a>
    </li>
    <li class="breadcrumb-item inline-block relative text-sm mx-[11px]">
        Proyectos
    </li>
</ol>
@endsection

@section('titleContent')
<h3 class="!mb-0">
    Listado de Proyectos
</h3>
@endsection

@section('bodyContent')
<div class="container mx-auto px-4 sm:px-6 lg:px-8 py-8">
    <div class="trezo-card p-5">
        <div class="flex justify-between items-center mb-4">
            <h2 class="text-xl font-bold text-gray-700">Proyectos Registrados</h2>
            <div class="flex items-center space-x-2">
                <button id="btn-nuevo" class="bg-blue-500 hover:bg-blue-700 text-white font-bold py-2 px-4 rounded-lg shadow-md transition duration-300 flex items-center">
                    <i class="material-symbols-outlined mr-2">add</i> Nuevo
                </button>
            </div>
        </div>

        <!-- Tabla de Proyectos -->
        <div class="overflow-x-auto bg-white rounded-lg shadow">
            <table id="proyectos-table" class="min-w-full leading-normal">
                <thead class="bg-gray-200 text-gray-600 uppercase text-sm">
                    <tr>
                        <th class="px-5 py-3 border-b-2 border-gray-200 text-left">ID</th>
                        <th class="px-5 py-3 border-b-2 border-gray-200 text-left">Nombre</th>
                        <th class="px-5 py-3 border-b-2 border-gray-200 text-left">Dirección</th>
                        <th class="px-5 py-3 border-b-2 border-gray-200 text-center">Acciones</th>
                    </tr>
                </thead>
                <tbody class="text-gray-700">
                    <!-- Los datos se cargarán aquí vía AJAX -->
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Modal para Crear/Editar Proyecto -->
<div id="proyecto-modal" class="fixed inset-0 bg-gray-900/60 bg-opacity-50 flex items-center justify-center z-50 hidden backdrop-blur-sm transition-opacity duration-300">
    <div class="bg-white rounded-2xl shadow-2xl w-full max-w-2xl transform transition-all scale-95 p-0 overflow-hidden">
        
        <!-- Modal Header -->
        <div class="bg-gray-50 px-6 py-4 border-b border-gray-100 flex justify-between items-center">
            <h2 id="modal-title" class="text-xl font-bold text-gray-800"></h2>
            <button type="button" id="btn-close-modal" class="text-gray-400 hover:text-gray-600 transition-colors">
                <span class="material-symbols-outlined">close</span>
            </button>
        </div>

        <form id="proyecto-form" class="p-6">
            @csrf
            <input type="hidden" id="id" name="id">

            <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                
                <!-- Nombre -->
                <div class="col-span-2">
                    <label for="nombre" class="block text-sm font-semibold text-gray-700 mb-1">Nombre del Proyecto</label>
                    <div class="relative">
                        <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                            <span class="material-symbols-outlined text-gray-400">business_center</span>
                        </div>
                        <input type="text" id="nombre" name="nombre" 
                            class="pl-10 w-full rounded-lg border-gray-200 bg-gray-50 focus:bg-white focus:border-blue-500 focus:ring-blue-500 text-gray-800 text-sm py-2.5 transition-all outline-none" 
                            placeholder="Ej. Fibra Alemania - Fase 1" required>
                    </div>
                    <p id="error-nombre" class="text-red-500 text-xs mt-1 hidden"></p>
                </div>

                <!-- Dirección -->
                <div class="col-span-2">
                    <label for="direccion" class="block text-sm font-semibold text-gray-700 mb-1">Dirección</label>
                    <div class="relative">
                         <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                            <span class="material-symbols-outlined text-gray-400">location_on</span>
                        </div>
                        <input type="text" id="direccion" name="direccion" 
                            class="pl-10 w-full rounded-lg border-gray-200 bg-gray-50 focus:bg-white focus:border-blue-500 focus:ring-blue-500 text-gray-800 text-sm py-2.5 transition-all outline-none"
                            placeholder="Av. Principal 123">
                    </div>
                    <p id="error-direccion" class="text-red-500 text-xs mt-1 hidden"></p>
                </div>

                <!-- Teléfono -->
                <div>
                    <label for="telefono" class="block text-sm font-semibold text-gray-700 mb-1">Teléfono</label>
                    <div class="relative">
                        <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                            <span class="material-symbols-outlined text-gray-400">phone</span>
                        </div>
                        <input type="text" id="telefono" name="telefono" 
                            class="pl-10 w-full rounded-lg border-gray-200 bg-gray-50 focus:bg-white focus:border-blue-500 focus:ring-blue-500 text-gray-800 text-sm py-2.5 transition-all outline-none"
                            placeholder="+51 999 999 999">
                    </div>
                    <p id="error-telefono" class="text-red-500 text-xs mt-1 hidden"></p>
                </div>

                <!-- Email -->
                <div>
                    <label for="email" class="block text-sm font-semibold text-gray-700 mb-1">Email</label>
                    <div class="relative">
                        <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                            <span class="material-symbols-outlined text-gray-400">email</span>
                        </div>
                        <input type="email" id="email" name="email" 
                            class="pl-10 w-full rounded-lg border-gray-200 bg-gray-50 focus:bg-white focus:border-blue-500 focus:ring-blue-500 text-gray-800 text-sm py-2.5 transition-all outline-none"
                            placeholder="contacto@empresa.com">
                    </div>
                    <p id="error-email" class="text-red-500 text-xs mt-1 hidden"></p>
                </div>

                <!-- Sitio Web -->
                <div>
                    <label for="website" class="block text-sm font-semibold text-gray-700 mb-1">Sitio Web</label>
                    <div class="relative">
                        <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                            <span class="material-symbols-outlined text-gray-400">language</span>
                        </div>
                        <input type="text" id="website" name="website" 
                            class="pl-10 w-full rounded-lg border-gray-200 bg-gray-50 focus:bg-white focus:border-blue-500 focus:ring-blue-500 text-gray-800 text-sm py-2.5 transition-all outline-none"
                            placeholder="www.empresa.com">
                    </div>
                    <p id="error-website" class="text-red-500 text-xs mt-1 hidden"></p>
                </div>

                <!-- Logo -->
                <div>
                    <label for="logo_compania" class="block text-sm font-semibold text-gray-700 mb-1">Logo URL</label>
                    <div class="relative">
                        <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                            <span class="material-symbols-outlined text-gray-400">image</span>
                        </div>
                        <input type="text" id="logo_compania" name="logo_compania" 
                            class="pl-10 w-full rounded-lg border-gray-200 bg-gray-50 focus:bg-white focus:border-blue-500 focus:ring-blue-500 text-gray-800 text-sm py-2.5 transition-all outline-none"
                            placeholder="https://...">
                    </div>
                    <p id="error-logo_compania" class="text-red-500 text-xs mt-1 hidden"></p>
                </div>
            </div>

            <div class="mt-8 flex justify-end gap-3">
                <button type="button" id="btn-cancelar" class="px-5 py-2.5 rounded-lg border border-gray-200 text-gray-600 hover:bg-gray-50 font-medium transition-colors text-sm">
                    Cancelar
                </button>
                <button type="submit" id="btn-guardar" class="px-5 py-2.5 rounded-lg bg-blue-600 text-white hover:bg-blue-700 font-medium shadow-md hover:shadow-lg transition-all text-sm flex items-center">
                    <span class="material-symbols-outlined text-lg mr-2">save</span> Guardar
                </button>
            </div>
        </form>
    </div>
</div>
@endsection

@section('scriptsJS')

<script src="https://cdn.datatables.net/2.3.5/js/dataTables.js"></script>
<script src="https://cdn.datatables.net/2.3.5/js/dataTables.tailwindcss.js"></script>
<script src="{{ asset('js/CRUD.js') }}"></script>

<script>
$(document).ready(function() {
    let isEditing = false;
    let currentProyectoId = null;
    const proyectoUrl = "{{ url('/rproyecto') }}";

    const table = $('#proyectos-table').DataTable({
        processing: true,
        serverSide: true,
        ajax: "{{ route('rproyecto.index') }}",
        columns: [
            { data: 'id', name: 'id' },
            { data: 'nombre', name: 'nombre' },
            { data: 'direccion', name: 'direccion' },
            { data: 'action', name: 'action', orderable: false, searchable: false, className: 'text-center' }
        ],
        dom: '<"flex justify-between items-center mb-4"lf><"overflow-x-auto"rt><"flex justify-between items-center mt-4"ip>',
        initComplete: function () {
            const wrapper = this.api().table().container().closest('.dataTables_wrapper');

            // Estilizar input de búsqueda
            $(wrapper).find('div.dataTables_filter input')
                .addClass('px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-1 focus:ring-blue-400 focus:border-blue-400 text-sm')
                .attr('placeholder', '🔍 Buscar...');

            // Estilizar selector de filas
            $(wrapper).find('div.dataTables_length select')
                .addClass('px-2 py-1 border border-gray-300 rounded-md text-sm bg-white');
        },
        drawCallback: function () {
            const wrapper = this.api().table().container().closest('.dataTables_wrapper');
            const paginate = $(wrapper).find('div.dataTables_paginate');

            // Estilos para los botones de paginación
            paginate.find('a.paginate_button').removeClass().addClass(
                'px-3 py-1 border border-gray-300 rounded-md text-sm transition-colors duration-200 cursor-pointer bg-gray-50 text-gray-700 hover:bg-blue-100 hover:text-blue-800'
            );

            // Estilo para el botón de página activa
            paginate.find('a.paginate_button.current')
                .removeClass('bg-gray-50 text-gray-700 hover:bg-blue-100 hover:text-blue-800').addClass(
                    'px-3 py-1 border rounded-md text-sm bg-blue-600 text-white border-blue-600 cursor-pointer'
            );

            // Estilo para el botón deshabilitado
            paginate.find('a.paginate_button.disabled').addClass('opacity-50 cursor-not-allowed');

            $('#proyectos-table tbody tr td').addClass('py-3 px-5 border-b border-gray-200 whitespace-nowrap');
        }
    });

    // --- MANEJO DEL MODAL ---
    function openModal() {
        $('#proyecto-modal').removeClass('hidden');
        setTimeout(() => $('#proyecto-modal > div').removeClass('scale-95'), 50);
    }

    function closeModal() {
        $('#proyecto-modal > div').addClass('scale-95');
        setTimeout(() => {
            $('#proyecto-modal').addClass('hidden');
            resetForm(); // Reseteamos el estado completo al cerrar
        }, 150);
    }

    function resetForm() {
        // Reseteamos las variables de estado
        $('#proyecto-form')[0].reset();
        currentProyectoId = null;
        isEditing = false;
        $('.text-red-500').text('').addClass('hidden');
        $('#proyecto-form input').removeClass('border-red-500');
    }

    // --- ABRIR MODAL (NUEVO) ---
    $('#btn-nuevo').on('click', function() {
        resetForm();
        isEditing = false; // Aseguramos el estado para "nuevo"
        currentProyectoId = null;
        $('#modal-title').text('Nuevo Proyecto');
        
        // Habilitar campos y mostrar botón guardar
        $('#proyecto-form input').prop('disabled', false);
        $('#btn-guardar').removeClass('hidden');
        
        openModal();
    });

    // --- ABRIR MODAL (VER) ---
    $('#proyectos-table tbody').on('click', '.btn-ver', function() {
        let id = $(this).data('id');
        
        // Limpiamos errores y reseteamos form
        $('#proyecto-form')[0].reset();
        $('.text-red-500').text('').addClass('hidden');
        $('#proyecto-form input').removeClass('border-red-500');

        CRUD.getById(proyectoUrl, id).then(function(response) {
            if (response.success) {
                let proyecto = response.data;
                $('#modal-title').text('Ver Proyecto');
                
                // Llenar campos
                $('#id').val(proyecto.id);
                $('#nombre').val(proyecto.nombre);
                $('#direccion').val(proyecto.direccion);
                $('#telefono').val(proyecto.telefono);
                $('#email').val(proyecto.email);
                $('#website').val(proyecto.website);
                $('#logo_compania').val(proyecto.logo_compania);

                // Deshabilitar campos y ocultar botón guardar
                $('#proyecto-form input').prop('disabled', true);
                $('#btn-guardar').addClass('hidden');
                
                openModal();
            }
        });
    });

    // --- ABRIR MODAL (EDITAR) ---
    $('#proyectos-table tbody').on('click', '.btn-editar', function() {
        let id = $(this).data('id');
        currentProyectoId = id;
        isEditing = true;

        // Limpiamos solo los errores visuales, no el estado
        $('#proyecto-form')[0].reset();
        $('.text-red-500').text('').addClass('hidden');
        $('#proyecto-form input').removeClass('border-red-500');
        
        // Habilitar campos y mostrar botón guardar (por si venimos de Ver)
        $('#proyecto-form input').prop('disabled', false);
        $('#btn-guardar').removeClass('hidden');

        CRUD.getById(proyectoUrl, id).then(function(response) {
            if (response.success) {
                let proyecto = response.data;
                $('#modal-title').text('Editar Proyecto');
                $('#id').val(proyecto.id);
                $('#nombre').val(proyecto.nombre);
                $('#direccion').val(proyecto.direccion);
                $('#telefono').val(proyecto.telefono);
                $('#email').val(proyecto.email);
                $('#website').val(proyecto.website);
                $('#logo_compania').val(proyecto.logo_compania);
                openModal();
            }
        });
    });

    // --- CERRAR MODAL ---
    $('#btn-cancelar, #btn-close-modal').on('click', closeModal);
    $('#proyecto-modal').on('click', function(e) {
        if (e.target.id === 'proyecto-modal') {
            closeModal();
        }
    });

    // --- GUARDAR / ACTUALIZAR ---
    $('#proyecto-form').on('submit', function(e) {
        e.preventDefault();
        $('#btn-guardar').prop('disabled', true).text('Guardando...');

        let formData = $(this).serializeArray();
        let promise;

        if (isEditing) {
            promise = CRUD.update(proyectoUrl, currentProyectoId, $.param(formData));
        } else {
            promise = CRUD.create(proyectoUrl, $.param(formData));
        }

        promise.then(function(response) {
            if (response.success) {
                closeModal();
                table.ajax.reload();
                Swal.fire({
                    icon: 'success',
                    title: '¡Éxito!',
                    text: response.message,
                    timer: 1500,
                    showConfirmButton: false
                });
            }
        }).fail(function(xhr) {
            if (xhr.status === 422) {
                let errors = xhr.responseJSON.errors;
                $('.text-red-500').text('').addClass('hidden');
                $('#proyecto-form input').removeClass('border-red-500');

                for (const key in errors) {
                    $(`#${key}`).addClass('border-red-500');
                    $(`#error-${key}`).text(errors[key][0]).removeClass('hidden');
                }
            } else {
                Swal.fire({
                    icon: 'error',
                    title: 'Error',
                    text: 'Ocurrió un error inesperado.'
                });
            }
        }).always(function() {
            $('#btn-guardar').prop('disabled', false).text('Guardar');
        });
    });

    // --- ELIMINAR ---
    $('#proyectos-table tbody').on('click', '.btn-eliminar', function() {
        let id = $(this).data('id');

        Swal.fire({
            title: '¿Estás seguro?',
            text: "¡No podrás revertir esta acción!",
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#3085d6',
            cancelButtonColor: '#d33',
            confirmButtonText: 'Sí, ¡eliminar!',
            cancelButtonText: 'Cancelar'
        }).then((result) => {
            if (result.isConfirmed) {
                CRUD.delete(proyectoUrl, id).then(function(response) {
                    if (response.success) {
                        table.ajax.reload();
                        Swal.fire(
                            '¡Eliminado!',
                            response.message,
                            'success'
                        );
                    }
                }).fail(function(xhr) {
                    let message = 'Ocurrió un error al eliminar.';
                    if (xhr.status === 409) { // Conflicto
                        message = xhr.responseJSON.message;
                    }
                    Swal.fire(
                        'Error',
                        message,
                        'error'
                    );
                });
            }
        });
    });
});
</script>
@endsection
